﻿' 版权所有 (C) Microsoft Corporation。保留所有权利。
''' <summary>
''' 此类表示方块的网格。它处理大多数游戏规则。
''' </summary>
''' <remarks></remarks>
Public Class Grid
    ' 网格是一个由 12 列和 15 行组成的 Block 对象。
    Dim matrix(11, 14) As Block

    ''' <summary>
    '''创建一些方块行以开始游戏。游戏从红色、蓝色和绿色方块开始。
    ''' </summary>
    ''' <param name="nrows">为开始游戏要创建的方块行数。</param>
    ''' <remarks></remarks>
    Public Sub New(ByVal nrows As Integer)
        If nrows > matrix.GetLength(0) Then
            Throw New Exception("Must start with " & matrix.GetLength(0) & " or fewer rows.")
        End If
        Dim row As Integer
        Dim column As Integer
        For row = 0 To nrows - 1
            For column = 0 To matrix.GetLength(1) - 1
                matrix(row, column) = New Block(New Color() {Color.Red, Color.Blue, Color.Green})
            Next
        Next
        For row = nrows To matrix.GetLength(0) - 1
            For column = 0 To matrix.GetLength(1) - 1
                matrix(row, column) = Nothing
            Next
        Next
    End Sub

    ''' <summary>
    ''' 新行可以在任何时候添加。新行除了红色、蓝色和绿色方块
    ''' 外，还有灰色方块。这使游戏变得更难。
    ''' </summary>
    ''' <remarks></remarks>
    Public Sub AddRow()
        Dim column As Integer
        ' 向每列添加一个新方块。
        For column = 0 To matrix.GetLength(1) - 1
            Dim newBlock As New Block(New Color() {Color.Red, Color.Blue, Color.Green, Color.Gray})
            ' 将新方块添加在列的底部，并将其余
            ' 的方块上推一列。
            For row As Integer = matrix.GetLength(0) - 1 To 1 Step -1
                matrix(row, column) = matrix(row - 1, column)
            Next
            matrix(0, column) = newBlock
        Next
    End Sub

    ''' <summary>
    ''' 绘制方块网格
    ''' </summary>
    ''' <param name="graphics"></param>
    ''' <param name="backColor"></param>
    ''' <remarks></remarks>
    Public Sub Draw(ByVal graphics As Graphics, ByVal backColor As Color)
        graphics.Clear(backColor)
        Dim row As Integer
        Dim column As Integer
        Dim theBlock As Block
        For row = 0 To matrix.GetLength(0) - 1
            For column = 0 To matrix.GetLength(1) - 1
                theBlock = matrix(row, column)
                If Not theBlock Is Nothing Then
                    Dim pointA As New Point(column * Block.BlockSize, row * Block.BlockSize)
                    matrix(row, column).Draw(graphics, pointA)
                End If
            Next
        Next
    End Sub

    ''' <summary>
    ''' 此方法响应 UI 中的单击事件。
    ''' </summary>
    ''' <param name="point"></param>
    ''' <returns>从网格中移除的方块数量。</returns>
    ''' <remarks></remarks>
    Public Function Click(ByVal point As Point) As Integer
        ' 计算出行和列。
        Dim total As Integer
        Dim transPt As Point = PointTranslator.TranslateToTL(point)
        Dim selectedRow As Integer = transPt.Y \ Block.BlockSize
        Dim selectedColumn As Integer = transPt.X \ Block.BlockSize
        Dim selectedBlock As Block = matrix(selectedRow, selectedColumn)
        If Not selectedBlock Is Nothing Then
            selectedBlock.MarkedForDeletion = True
            ' 判断是否有任何相邻的方块颜色相同。
            FindSameColorNeighbors(selectedRow, selectedColumn)
            ' 确定将消除多少个方块。
            total = Me.CalculateScore()
            If total > 1 Then
                Me.CollapseBlocks()
            Else
                Me.ClearMarkedForDeletion()
            End If
        End If
        Return total
    End Function

    Private Sub ClearMarkedForDeletion()
        Dim row As Integer
        Dim column As Integer
        For column = matrix.GetLength(1) - 1 To 0 Step -1
            ' 如果列完全为空，则将所有内容下移一列。
            For row = 0 To matrix.GetLength(0) - 1
                If Not matrix(row, column) Is Nothing Then
                    matrix(row, column).MarkedForDeletion = False
                End If
            Next
        Next
    End Sub

    ''' <summary>
    ''' 确定将消除多少个方块。
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Private Function CalculateScore() As Integer
        Dim row As Integer
        Dim column As Integer
        Dim total As Integer = 0
        For column = matrix.GetLength(1) - 1 To 0 Step -1
            ' 如果列完全为空，则将所有内容下移一列。
            For row = 0 To matrix.GetLength(0) - 1
                If Not matrix(row, column) Is Nothing Then
                    If matrix(row, column).MarkedForDeletion Then
                        total += 1
                    End If
                End If
            Next
        Next
        Return total
    End Function
    ''' <summary>
    ''' 在从列中移除方块之后，可能会出现
    ''' 空列。将列从右向左移动
    ''' 以填充空列。
    ''' </summary>
    ''' <remarks></remarks>
    Public Sub CollapseColumns()
        Dim row As Integer
        Dim column As Integer
        For column = matrix.GetLength(1) - 1 To 0 Step -1
            ' 如果列完全为空，则所有列上移一列。
            Dim noBlocks As Boolean = True
            For row = 0 To matrix.GetLength(0) - 1
                If Not matrix(row, column) Is Nothing Then
                    noBlocks = False
                End If
            Next

            If noBlocks Then
                Dim newcol As Integer
                For newcol = column To matrix.GetLength(1) - 2
                    For row = 0 To matrix.GetLength(0) - 1
                        matrix(row, newcol) = matrix(row, newcol + 1)
                    Next
                Next
                newcol = matrix.GetLength(1) - 1
                For row = 0 To matrix.GetLength(0) - 1
                    matrix(row, newcol) = Nothing
                Next
            End If
        Next

    End Sub

    ''' <summary>
    ''' 从网格中移除所有方块。
    ''' </summary>
    ''' <remarks></remarks>
    Public Sub CollapseBlocks()
        Dim theBlock As Block
        Dim column As Integer
        Dim row As Integer
        Dim aRow As Integer

        ' 首先移除每列中的方块。
        For column = 0 To matrix.GetLength(1) - 1
            For row = matrix.GetLength(0) - 1 To 0 Step -1
                theBlock = matrix(row, column)
                If (Not theBlock Is Nothing) Then
                    If theBlock.MarkedForDeletion Then
                        For aRow = row To matrix.GetLength(0) - 2
                            matrix(aRow, column) = matrix(aRow + 1, column)
                        Next
                        matrix(matrix.GetLength(0) - 1, column) = Nothing
                    End If
                End If
            Next
        Next

        ' 重置 MarkedForDeletion 标志。
        For row = 0 To matrix.GetLength(0) - 1
            For column = 0 To matrix.GetLength(1) - 1
                theBlock = matrix(row, column)
                If Not theBlock Is Nothing Then
                    theBlock.MarkedForDeletion = False
                End If
            Next
        Next

        ' 移除现在为空的任何列。
        CollapseColumns()
    End Sub

    ''' <summary>
    ''' 提供对网格的访问。
    ''' </summary>
    ''' <param name="row"></param>
    ''' <param name="column"></param>
    ''' <value></value>
    ''' <remarks></remarks>
    Default Public Property Item(ByVal row As Integer, ByVal column As Integer) As Block
        Get
            Return matrix(row, column)
        End Get
        Set(ByVal Value As Block)
            matrix(row, column) = Value
        End Set
    End Property


    Private blocksToExamine As ArrayList
    ''' <summary>
    ''' 将具有相同颜色的每个邻近方块的 MarkedForDeletion
    ''' 设置为 True
    ''' </summary>
    ''' <param name="row"></param>
    ''' <param name="column"></param>
    ''' <remarks></remarks>
    Private Sub FindSameColorNeighbors(ByVal row As Integer, ByVal column As Integer)
        Dim color As Color = matrix(row, column).Color
        blocksToExamine = New ArrayList
        blocksToExamine.Add(New Point(row, column))
        matrix(row, column).MarkedForDeletion = True

        ' 每当找到一个邻近方块时，标记它以便删除，
        ' 并将它添加到方块列表以查找邻近方块。检查完
        ' 它之后，将它从列表中移除。重复此操作，
        ' 直至没有可检查的方块为止。
        While blocksToExamine.Count > 0
            FindNeighbors()
        End While
    End Sub

    ''' <summary>
    ''' 查看每一边的方块。
    ''' </summary>
    ''' <remarks></remarks>
    Private Sub FindNeighbors()
        ' 取出 arraylist 的第一个方块并检查它。
        Dim location As Point = CType(blocksToExamine(0), Point)
        Dim currentBlock As Block = matrix(location.X, location.Y)
        Dim row As Integer = location.X
        Dim column As Integer = location.Y
        blocksToExamine.RemoveAt(0)

        Dim nextRow As Integer
        Dim nextCol As Integer
        Dim selected As Block

        ' 向上检查
        If row < matrix.GetLength(0) - 1 Then
            nextRow = row + 1
            selected = matrix(nextRow, column)
            ExamineNeighbor(selected, nextRow, column, currentBlock.Color)
        End If

        ' 向下检查
        If row > 0 Then
            nextRow = row - 1
            selected = matrix(nextRow, column)
            ExamineNeighbor(selected, nextRow, column, currentBlock.Color)
        End If


        ' 向左检查
        If column > 0 Then
            nextCol = column - 1
            selected = matrix(row, nextCol)
            ExamineNeighbor(selected, row, nextCol, currentBlock.Color)
        End If

        ' 向右检查
        If column < matrix.GetLength(1) - 1 Then
            nextCol = column + 1
            selected = matrix(row, nextCol)
            ExamineNeighbor(selected, row, nextCol, currentBlock.Color)
        End If
    End Sub

    ''' <summary>
    ''' 如果邻近方块具有相同颜色，则将它添加到
    ''' 要检查的方块中。
    ''' </summary>
    ''' <param name="selected"></param>
    ''' <param name="row"></param>
    ''' <param name="column"></param>
    ''' <param name="color"></param>
    ''' <remarks></remarks>
    Private Sub ExamineNeighbor(ByVal selected As Block, ByVal row As Integer, ByVal column As Integer, ByVal color As Color)
        If Not selected Is Nothing Then
            If selected.Color.Equals(color) Then
                If Not selected.MarkedForDeletion Then
                    selected.MarkedForDeletion = True
                    blocksToExamine.Add(New Point(row, column))
                End If
            End If
        End If
    End Sub



End Class

